package modbus;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.net.Socket;

//import bacnet.bac_def;

public class mod
{
    public class def
    {
        // Maximum number of modbus registers you can read or write
        // in one operation.
        //
        public final static int  MOD_MAX_REG              = 32;
        public final static int  MOD_MAX_BYTE             = 512;
        //
        // Modbus function numbers.
        //
        public final static byte MOD_DLE                  = 0x10;
        public final static byte MOD_STX                  = 2;
        public final static byte MOD_ETX                  = 3;
        //
        public final static int  PROT_ASCII               = 0;
        public final static int  PROT_RTU                 = 1;
        public final static int  PROT_DF1                 = 2;
        public final static int  PROT_TCP                 = 3;
        //
        public final static int  PORT_DF1                 = 5001;
        public final static int  PORT_TCP                 = 502;
        public final static int  PORT_RCLI                = 5003;
        //
        public final static int  SIZE_16                  = 1;
        public final static int  SIZE_STR                 = 2;
        //
        public final static int  READ_COILS               = 1;
        public final static int  READ_DISCRETE_INPUTS     = 2;
        public final static int  READ_HOLDING_REGISTERS   = 3;
        public final static int  READ_INPUT_REGISTERS     = 4;
        public final static int  WRITE_SINGLE_COIL        = 5;
        public final static int  WRITE_SINGLE_REGISTER    = 6;
        public final static int  READ_EXCEPTION_STATUS    = 7;
        public final static int  FETCH_COMM_EVENT_CTR     = 11;
        public final static int  FETCH_COMM_EVENT_LOG     = 12;
        public final static int  WRITE_MULTIPLE_COILS     = 15;
        public final static int  WRITE_MULTIPLE_REGISTERS = 16;
        public final static int  REPORT_SLAVE_ID          = 17;
        public final static int  READ_GENERAL_REFERENCE   = 20;
        public final static int  WRITE_GENERAL_REFERENCE  = 21;
        public final static int  MASK_WRITE_4X_REGISTER   = 22;
        public final static int  READ_WRITE_4X_REGISTERS  = 23;
        public final static int  READ_FIFO_QUEUE          = 24;
        //
        // ---Extended functions added by RCW---
        //
        // Notation for extended functions.
        // All values are exchanged big-endian.
        //
        // S = Slave address.
        // F = Function.
        // AA = Register address.
        //
        // NU = Not used, encoded as two zero bytes.
        // QQ = Number of registers.
        // CC = Number of data bytes following in request/response.
        // WW = 16-bit value.
        // DDDD = 32-bit value.
        // FFFF = Floating point value.
        // c..c\0 = Null terminated ascii string.
        // b..b = Binary data.
        //
        public static final int  RD_REGS_16               = 3;
        // Req : S F AA QQ
        // Rsp : S F AA QQ WW WW ...
        public static final int  WR_REGS_16               = 16;
        // Req : S F AA QQ WW WW ...
        // Rsp : S F AA QQ
        public static final int  RD_REGS_STR              = 100;
        // Req : S F AA QQ
        // Rsp : S F AA QQ c..c\0c..c\0...
        public static final int  WR_REGS_STR              = 101;
        // Req : S F AA QQ c..c\0c..c\0...
        // Rsp : S F AA QQ
        public static final int  RD_REGS_LIST             = 102;
        // Req : S F NU QQ AA AA AA ...
        // Rsp : S F NU QQ c..c\0c..c\0...
        public static final int  WR_REGS_LIST             = 103;
        // Req : S F NU QQ AAc..c\0AAc..c\0...
        // Req : S F NU QQ
        //
        // This function is used to relay a message exactly as received
        // to a Modbus compatible device on a serial bus.
        //
        public static final int  RELAY_MESSAGE            = 104;
        // Req : S F NU CC ADU
        // Req : S F NU CC ADU
        //
        // ENTRY:
        // S = node ID of relay unit.
        // F = RELAY_MESSAGE.
        // NU = '00'
        // CC = number of bytes in request ADU.
        // ADU = Modbus request ADU.
        //
        // RETURN:
        // S = unchanged.
        // F = unchanged or |= 0x80 if timeout.
        // NU = '00'
        // CC = number of bytes in response ADU.
        // ADU = Modbus response ADU.
        //
        // This function is used to perform Remote Procedure Calls (RPC).
        // Parameters passed and data returned are all 32-bit values, signed or
        // unsigned as needed by the application.
        // If 'QQ' is a negative number, the data is interpreted as a binary
        // data block, and is not broken up into 4-byte long values.
        //
        public static final int  MRPC_FCN                 = 105;
        // Req : S F AA QQ DDDD DDDD ...
        // Rsp : S F AA QQ DDDD DDDD ...
        //
        // ENTRY:
        // S = Slave address.
        // F = MRPC_FCN.
        // AA = Function type.
        // QQ = Number of 32-bit request parameters.
        // DDDD = 32-bit request values (or binary data).
        //
        // RETURN:
        // S = unchanged.
        // F = unchanged or |= 0x80 if error.
        // AA = Function type.
        // QQ = Number of 32-bit response parameters.
        // DDDD = 32-bit response values (or binary data).
        //
        // Functions used to read/write BACnet object properties.
        // Object properties are always read and written as strings.
        // Does not support all object properties.
        // Object property is never 0; '00' used to terminate property
        // list for read operation (TT).
        // Write operations return a single byte return code for each
        // property written. May be omitted if all operations succeed.
        // Return code will be '0' if success.
        //
        // QQ = Property Count and Index Flag:
        // Count is unsigned, with MSBit acting as Index Flag.
        // If MSBit of count is '0', IIII is object id as described below.
        // If MSBit of count is '1', IIII is index of object in server object
        // array.
        // IIII = Object id; (BACnetObjectType (Db.h) << 22) + object instance.
        // PP = BACnet object property (vsbhp.h).
        // C = Property count (number of properties read/written).
        // R = Return code if error return:
        // ILLEGAL_OBJECT_INSTANCE
        // ILLEGAL_DATA_ADDRESS
        // ILLEGAL_DATA_VALUE
        //
        public static final int  RD_OBJ                   = 106;
        // Req : S F QQ IIII PP PP...
        // Rsp : S F C c..c\0 c..c\0... (Success)
        // Rsp : S F R (Failure)

        public static final int  WR_OBJ                   = 107;
        // Req : S F QQ IIII PP c..c\0 PP c..c\0...
        // Rsp : S F C (Success)
        // Rsp : S F R (Failure)

        public static final int  RD_DEV_REGS_STR          = 108;
        // Req : S F AA QQ
        // Rsp : S F AA QQ c..c\0c..c\0...
        public static final int  WR_DEV_REGS_STR          = 109;
        // Req : S F AA QQ c..c\0c..c\0...
        // Rsp : S F AA QQ
        public static final int  RD_DEV_REGS_LIST         = 110;
        // Req : S F NU QQ AA AA AA ...
        // Rsp : S F NU QQ c..c\0c..c\0...
        public static final int  WR_DEV_REGS_LIST         = 111;
        // Req : S F NU QQ AAc..c\0AAc..c\0...
        // Req : S F NU QQ
        public static final int  RD_FILE                  = 112;
        // Req : S F QQ IIII PP PP...
        // Rsp : S F C c..c\0 c..c\0... (Success)
        // Rsp : S F R (Failure)
        public static final int  WR_FILE                  = 113;
        // Req : S F QQ IIII PP PP...
        // Rsp : S F C c..c\0 c..c\0... (Success)
        // Rsp : S F R (Failure)
        public static final int  DEL_FILE                  = 114;
        // Req : S F QQ IIII PP PP...
        // Rsp : S F C c..c\0 c..c\0... (Success)
        // Rsp : S F R (Failure)
        //
        // Modbus error codes.
        //
        public final static byte UNKNOWN_ERROR            = 0;
        public final static byte ILLEGAL_FUNCTION         = 1;
        public final static byte ILLEGAL_DATA_ADDRESS     = 2;
        public final static byte ILLEGAL_DATA_VALUE       = 3;
        public final static byte SLAVE_DEVICE_FAILURE     = 4;
        public final static byte ACKNOWLEDGE              = 5;
        public final static byte SLAVE_DEVICE_BUSY        = 6;
        public final static byte NEGATIVE_ACKNOWLEDGE     = 7;
        public final static byte MEMORY_PARITY_ERROR      = 8;
        //
        public final static byte GATEWAY_NO_PATH          = 10;
        public final static byte GATEWAY_TIMEOUT          = 11;
        //
        public final static byte RX_PACKET_CORRUPT        = 12;
        public final static byte DATA_RX_ERROR            = 13;
        public final static byte DATA_TX_ERROR            = 14;
        public final static byte ILLEGAL_OBJECT_INSTANCE  = 15;
        //
        //Modbus device type
        //
        public final static byte BAS_STAT            	= 0;
        public final static byte BELIMO_UK24MOD         = 1;
        public final static byte CIRCUTOR_CVM_NRG96     = 2;
        public final static byte HONEYWELL_S7810M       = 3;
        public final static byte NORTHERN_DESIGN_350    = 4;
        public final static byte RENU_FIOD_1600         = 5;
        public final static byte RENU_FIOD_0008        	= 6;
        public final static byte RENU_FIOD_0808     	= 7;
        public final static byte SCHNEIDER_PL800    	= 8;
        public final static byte VERIS_H8035          	= 10;
        public final static byte WNC_3Y_208_MB          = 11;
        public final static byte WNC_3Y_400_MB          = 12;
        public final static byte WNC_3Y_480_MB          = 13;
        public final static byte WNC_3Y_600_MB          = 14;
        public final static byte WNC_3D_240_MB          = 15;
        public final static byte WNC_3D_400_MB          = 16;
        public final static byte WNC_3D_480_MB          = 17;
    }

    public enum err
    {
        UNKNOWN_ERROR( "Unknown Error" , def.UNKNOWN_ERROR ) ,
        //
        ILLEGAL_FUNCTION( "Illegal Function" , def.ILLEGAL_FUNCTION ) ,
        ILLEGAL_DATA_ADDRESS( "Illegal Data Address" , def.ILLEGAL_DATA_ADDRESS ) ,
        ILLEGAL_DATA_VALUE( "Illegal Data Value" , def.ILLEGAL_DATA_VALUE ) ,
        SLAVE_DEVICE_FAILURE( "Slave Device Failure" , def.SLAVE_DEVICE_FAILURE ) ,
        ACKNOWLEDGE( "Acknowledge" , def.ACKNOWLEDGE ) ,
        SLAVE_DEVICE_BUSY( "Slave Device Busy" , def.SLAVE_DEVICE_BUSY ) ,
        NEGATIVE_ACKNOWLEDGE( "Negative Acknowledge" , def.NEGATIVE_ACKNOWLEDGE ) ,
        MEMORY_PARITY_ERROR( "Memory Parity Error" , def.MEMORY_PARITY_ERROR ) ,
        //
        GATEWAY_NO_PATH( "Gateway No Path" , def.GATEWAY_NO_PATH ) ,
        GATEWAY_TIMEOUT( "Gateway Timeout" , def.GATEWAY_TIMEOUT ) ,
        //
        RX_PACKET_CORRUPT( "RX Packet Corrupt" , def.RX_PACKET_CORRUPT ) ,
        DATA_RX_ERROR( "RX Data Error" , def.DATA_RX_ERROR ) ,
        DATA_TX_ERROR( "TX Data Error" , def.DATA_TX_ERROR ) ,
        ILLEGAL_OBJECT_INSTANCE( "Illegal Object Instance" , def.ILLEGAL_OBJECT_INSTANCE ) , ;

        private String enum_txt;
        private int    enum_val;

        err( String txt , int val )
        {
            enum_txt = txt;
            enum_val = val;
        }

        public int getEnum()
        {
            return enum_val;
        }

        public String getText()
        {
            return enum_txt;
        }

        public static String getName( int eval )
        {
            for (err e : err.values())
            {
                if (e.enum_val == eval)
                    return e.name();
            }
            return err.UNKNOWN_ERROR.name();
        }

        public static err getObj( String val )
        {
            try
            {
                return err.valueOf(val);
            } catch (IllegalArgumentException e)
            {
                return err.UNKNOWN_ERROR;
            }
        }
    }

    private static class packet
    {
        // MBAP Header.
        public short  tran_id;
        public short  prot_id;
        public short  mbap_len;
        // Original Modbus frame.
        public byte   slv_adr;
        public byte   fcn;
        public short  itm_adr;
        public short  itm_qty;
        public byte   b[];
        public short  w[];
        public int    l[];
        public float  f[];
        public String s[];

        /** Creates a new instance of packet */
        public packet()
        {
            b = new byte[def.MOD_MAX_BYTE];
            w = new short[def.MOD_MAX_REG * 2];
            l = new int[def.MOD_MAX_REG];
            f = new float[def.MOD_MAX_REG];
            s = new String[def.MOD_MAX_REG];
        }
    }

    private static class transfer
    {
        final static int        MASK8          = 0x000000ff;
        final static int        MASK16         = 0x0000ffff;
        //
        int                     BUFFER_SIZE    = 512;
        //
        byte                    protocol       = def.PROT_TCP;
        //
        int                     last_char;
        int                     rx_length;
        int                     adu_length;
        //
        byte                    rxBuf[];                      // Input packets.
        byte                    xxBuf[];                      // Temporary.
        byte                    txBuf[];                      // Output
        // packets.
        //
        InputStream             dis            = null;
        OutputStream            dos            = null;

        /**
         * Converts the binary packet in src[] to a transmit
         * frame in dst[] according to the current protocol.
         * For standard Modbus/TCP, this is just an array copy;
         * for Modbus/DF1, it adds the DLE/STX start of packet,
         * the DLE/DLE's for values of 0x10 (DLE), and the DLE/ETX
         * end of packet.
         * @param length = number of bytes in src[]
         * @param src = received packet
         * @param dst = transmit frame
         * @return Number of bytes to transmit in dst[]
         */
        private int BinToTx( int length , byte src[] , byte dst[] )
        {
            int i , src_idx , dst_idx;
            int frame_length = 0;
            byte dchar;

            // Zero indexes.
            src_idx = 0;
            dst_idx = 0;
            // Which protocol?
            switch (protocol)
            {
                case def.PROT_DF1:
                    // Store leading DLE/STX.
                    dst[dst_idx++] = def.MOD_DLE;
                    dst[dst_idx++] = def.MOD_STX;
                    // Transfer the rest of the packet.
                    for (i = length; i > 0; --i)
                    {
                        // Get character.
                        dchar = src[src_idx++];
                        // If this is a DLE.
                        if (dchar == def.MOD_DLE)
                        {
                            // Add a DLE.
                            dst[dst_idx++] = dchar;
                            // Increment length.
                            ++frame_length;
                        }
                        // Store character.
                        dst[dst_idx++] = dchar;
                        // Increment length.
                        ++frame_length;
                    }
                    // Store trailing DLE/ETX.
                    dst[dst_idx++] = def.MOD_DLE;
                    dst[dst_idx++] = def.MOD_ETX;
                    // Adjust frame length and return.
                    frame_length += 4;
                    break;
                case def.PROT_TCP:
                    // Transfer binary data frame.
                    System.arraycopy(src, 0, dst, 0, length);
                    // Set frame length.
                    frame_length = length;
                    break;
                default:
                    // NOTREACHED.
                    break;
            }
            // And return.
            return frame_length;
        }

        /**
         * Converts the received frame in src[] to a binary packet
         * in dst[] according to the current protocol.
         * For standard Modbus/TCP, this is just an array copy;
         * for Modbus/DF1, it strips the DLE/STX start of packet,
         * the DLE/DLE's for values of 0x10 (DLE), and the DLE/ETX
         * end of packet.
         * @param length = number of bytes in the received packet
         * @param src = received packet
         * @param dst = binary packet
         * @return Number of bytes in dst[] if succes. Returns -1 if:
         * recevied frame doesn't start with DLE/STX, received frame
         * has unknown DLE/XXX sequence, received frame doesn't end
         * with DLE/ETX, and if received frame has more characters
         * after a DLE/ETX.
         */
        private int RxToBin( int length , byte src[] , byte dst[] )
        {
            int i , src_idx , dst_idx;
            int end_of_packet;
            int frame_length;
            byte this_chr;
            byte last_chr;

            // Keep Java from bitching.
            frame_length = 0;
            // Which protocol?
            switch (protocol)
            {
                case def.PROT_DF1:
                    // Error if first two characters aren't DLE STX.
                    if (src[0] != def.MOD_DLE)
                        return -1;
                    if (src[1] != def.MOD_STX)
                        return -1;
                    // Initialize array indexes.
                    src_idx = 2;
                    dst_idx = 0;
                    // Adjust receive length; remove leading DLE/STX.
                    length -= 2;
                    // Reset end-of-packet flag.
                    end_of_packet = 0;
                    // Reset last char..
                    last_chr = 0;
                    // Transfer data and remove DLE pairs.
                    i = length;
                    while (i-- > 0)
                    {
                        // Get a character.
                        this_chr = src[src_idx++];
                        // If last char was DLE.
                        if (last_chr == def.MOD_DLE)
                        {
                            // If this one is also a DLE.
                            if (this_chr == def.MOD_DLE)
                            {
                                // Put one DLE in dest.
                                dst[dst_idx++] = def.MOD_DLE;
                                // Decrement receive counter.
                                --length;
                                // Reset last char.
                                last_chr = 0;
                                // And continue;
                                continue;
                            }
                            // Else if this is the end.
                            else if (this_chr == def.MOD_ETX)
                            {
                                // Error if this isn't the last character.
                                if (i != 0)
                                    return -1;
                                // Set end-of-packet flag.
                                end_of_packet = 1;
                                // And continue.
                                continue;
                            }
                            // Else unknown sequence.
                            else
                                return -1;
                        }
                        // If this character is a DLE
                        if (this_chr == def.MOD_DLE)
                        {
                            // Make it the last character received.
                            last_chr = def.MOD_DLE;
                            // And continue with loop.
                            continue;
                        }
                        // Transfer character.
                        dst[dst_idx++] = this_chr;
                        // Make last char.
                        last_chr = this_chr;
                    }
                    // Error if end-of-packet not found.
                    if (end_of_packet == 0)
                        return -1;
                    // Return with real packet length; adjust for trailing DLE
                    // ETX.
                    frame_length = length - 2;
                    break;
                case def.PROT_TCP:
                    // Set frame length.
                    frame_length = length;
                    // Transfer binary data frame.
                    System.arraycopy(src, 0, dst, 0, length);
                    break;
                default:
                    // NOTREACHED.
                    break;
            }
            // And return.
            return frame_length;

        }

        /**
         * Reset receive members in preparation for receiving
         * a new packet.
         */
        private void ResetRx()
        {
            // Reset variables.
            last_char = 0;
            rx_length = 0;
            adu_length = 0;
        }

        /**
         * Adds a received character to the receive buffer according to the
         * current protocol.
         * @param dchar = character to add
         * @return Returns packet length if complete packet received,
         * '0' otherwise
         */
        private int RxChar( int dchar )
        {
            rxBuf[rx_length++] = (byte) dchar;
            // Which protocol?
            switch (protocol)
            {
                case def.PROT_DF1:
                    //
                    // NOTE: Added this code to make sure we start a
                    // packet with DLE/STX. This is a kluge.
                    //
                    // How many characters?
                    switch (rx_length)
                    {
                        case 1:
                            // If first character is not a DLE.
                            if (dchar != def.MOD_DLE)
                            {
                                // Reset the count.
                                rx_length = 0;
                                // Null the character.
                                dchar = 0;
                            }
                            break;
                        case 2:
                            // If second character is not a STX.
                            if (dchar != def.MOD_STX)
                            {
                                // Reset the count.
                                rx_length = 0;
                                // Null the character.
                                dchar = 0;
                            }
                            break;
                        default:
                            // If last character was a DLE.
                            if (last_char == def.MOD_DLE)
                            {
                                // And if the this character is also a DLE.
                                if (dchar == def.MOD_DLE)
                                {
                                    // Null this character, as we've received
                                    // two DLE's in a row. If we keep this as
                                    // last character, we could prematurely
                                    // end a packet if the next received char
                                    // is a '3'.
                                    dchar = 0;
                                }
                                // Else if current character is an ETX.
                                else if (dchar == def.MOD_ETX)
                                {
                                    // Packet is ready.
                                    return rx_length;
                                }
                            }
                            break;
                    }
                    break;
                case def.PROT_TCP:
                    // If we haven't gotten the adu_length yet.
                    if (adu_length == 0)
                    {
                        // If the length has been received.
                        if (rx_length == 6)
                        {
                            // Get the length field.
                            adu_length = (rxBuf[4] & 0x000000ff) << 8;
                            adu_length += rxBuf[5] & 0x000000ff;
                            adu_length += 6;
                        }
                    }
                    // Else if we've gotten all the characters.
                    else if (rx_length >= adu_length)
                    {
                        // Packet is ready.
                        return rx_length;
                    }
                    break;
                default:
                    // Should never get here.
                    break;
            }
            // Store last character.
            last_char = dchar;
            // And return.
            return 0;
        }

        /**
         * Read and discard all data in stream.
         * 
         * @throws IOException
         */
        private void DrainDis() throws IOException
        {
            while (dis.available() != 0)
            {
                try
                {
                    if (dis.read() == -1)
                        break;
                } catch (IOException e)
                {
                    throw e;
                }
            }
        }

        /**
         * Waits for an entire packet to be received. Data is stored in 'xxBuf'.
         * 
         * @return Number of bytes in packet.
         * @throws IOException
         */
        private int WaitPacket() throws IOException
        {
            boolean packet_found = false;
            int i , rx_count;

            // Get an entire packet, throw error if socket closed.
            try
            {
                rx_count = dis.read(xxBuf);
            } catch (IOException e)
            {
                throw e;
            }
            // Error if end of file; this will happen if server
            // terminates while still connected.
            if (rx_count < 0)
            {
                IOException e = new IOException(err.DATA_RX_ERROR.name());
                throw (e);
            }
            // Error if entire buffer filled (too many characters).
            if (rx_count == BUFFER_SIZE)
            {
                // Drain anything left in stream.
                try
                {
                    DrainDis();
                } catch (IOException e)
                {
                    throw e;
                }
                // Reset and throw exception.
                ResetRx();
                IOException e = new IOException(err.DATA_RX_ERROR.name());
                throw (e);
            }
            // Reset the receive system.
            ResetRx();
            // For each character.
            for (i = 0; i < rx_count; ++i)
            {
                // Non-zero return means packet found.
                if (RxChar(xxBuf[i]) != 0)
                {
                    packet_found = true;
                    break;
                }
            }
            // If we haven't found a packet (this should never happen).
            if (!packet_found)
            {
                // Drain anything left in stream.
                try
                {
                    DrainDis();
                } catch (IOException e)
                {
                    throw e;
                }
                // Reset and throw exception.
                ResetRx();
                IOException e = new IOException(err.DATA_RX_ERROR.name());
                throw (e);
            }
            // Return success.
            return rx_length;
        }

        /**
         * Here to convert a null terminated byte array to a String.
         * Returns the index of the start of the next string within the byte
         * array. Used to sequentially extract null terminated strings
         * within a rececived packet. 
         * @param src = byte array holding null terminated string(s)
         * @param b_idx = starting index within src[]
         * @param mex = Modbus packet; mod.packet
         * @param s_idx = index of String within mex.s[]
         * @return Returns the index of the start of the next string within src[]
         */
        private int BytesToString( byte[] src , int b_idx , packet mex , int s_idx )
        {
            byte dchar;

            // Start with null string.
            mex.s[s_idx] = "";
            // Loop until null terminator found.
            while (true)
            {
                // Get character, break if null.
                if ((dchar = src[b_idx++]) == 0)
                {
                    break;
                }
                // Add character to string.
                mex.s[s_idx] += Character.toString((char) dchar);
            }
            return b_idx;
        }

        //
        // ----------------------------------------------------------------------
        //
        // Here to convert a string to a null terminated byte array.
        // Returns with the index within dst[ ] of the next available byte for
        // the next string to be added.
        //
        private int StringToBytes( byte[] dst , int b_idx , packet mex , int s_idx )
        {
            int i , length;

            length = mex.s[s_idx].length();
            for (i = 0; i < length; ++i)
            {
                dst[b_idx++] = (byte) mex.s[s_idx].charAt(i);
            }
            dst[b_idx++] = 0;
            return b_idx;
        }

        /**
         * Master routine to read/write data items.
         * 
         * NOTE: 'itm_adr' is 1-based value; i.e. register 1 is passed as 1.
         * This routine converts the register to 0-based.
         * 
         * @param pkt = mod.packet object
         * @return 0 = success, throws Modbus exception code if error
         * @throws IOException (err.xxx.name()).
         */
        private int MasterTx( packet pkt ) throws IOException
        {
            int tmpval;
            int i , frame_length;
            int rx_idx , xx_idx;
            byte byte_val , byte_msk , byte_cnt;

            // Initialize indexes.
            rx_idx = xx_idx = 0;
            // If this is Modbus/TCP.
            if (protocol == def.PROT_TCP)
            {
                // Transaction id.
                xxBuf[xx_idx++] = (byte) (pkt.tran_id >> 8);
                xxBuf[xx_idx++] = (byte) pkt.tran_id;
                // Protocol id.
                xxBuf[xx_idx++] = (byte) (pkt.prot_id >> 8);
                xxBuf[xx_idx++] = (byte) pkt.prot_id;
                // Move past mbap_len.
                xx_idx += 2;
            }
            // Slave address.
            xxBuf[xx_idx++] = pkt.slv_adr;
            // Function.
            xxBuf[xx_idx++] = pkt.fcn;
            // Convert item address to 0-based, and store it big-endian.
            tmpval = pkt.itm_adr - 1;
            xxBuf[xx_idx++] = (byte) (tmpval >> 8);
            xxBuf[xx_idx++] = (byte) tmpval;
            // Convert and store rest of data.
            switch ((int) pkt.fcn)
            {
                case def.READ_COILS:
                case def.READ_DISCRETE_INPUTS:
                case def.READ_HOLDING_REGISTERS:
                case def.READ_INPUT_REGISTERS:
                    //
                    // Store item quantity.
                    xxBuf[xx_idx++] = (byte) (pkt.itm_qty >> 8);
                    xxBuf[xx_idx++] = (byte) pkt.itm_qty;
                    //
                    break;
                //
                case def.WRITE_SINGLE_COIL:
                    //
                    // Store data. '1' is 0xff00.
                    if (pkt.b[0] != 0)
                        tmpval = 0x0ff00;
                    else
                        tmpval = 0;
                    xxBuf[xx_idx++] = (byte) (tmpval >> 8);
                    xxBuf[xx_idx++] = (byte) tmpval;
                    //
                    break;
                //
                case def.WRITE_SINGLE_REGISTER:
                    //
                    // Store item data.
                    xxBuf[xx_idx++] = (byte) (pkt.w[0] >> 8);
                    xxBuf[xx_idx++] = (byte) pkt.w[0];
                    //
                    break;
                //
                case def.WRITE_MULTIPLE_COILS:
                    //
                    // Store item quantity.
                    xxBuf[xx_idx++] = (byte) (pkt.itm_qty >> 8);
                    xxBuf[xx_idx++] = (byte) pkt.itm_qty;
                    // Store number of bytes.
                    byte_cnt = (byte) (pkt.itm_qty / 8);
                    if ((pkt.itm_qty & 0x07) != 0)
                        ++byte_cnt;
                    xxBuf[xx_idx++] = (byte) byte_cnt;
                    // Store data.
                    byte_val = 0;
                    byte_msk = 1;
                    for (i = 0; i < pkt.itm_qty; ++i)
                    {
                        if (pkt.b[i] != 0)
                            byte_val |= byte_msk;
                        if (byte_msk == 0x80)
                        {
                            xxBuf[xx_idx++] = byte_val;
                            byte_val = 0;
                            byte_msk = 1;
                            --byte_cnt;
                        } else
                            byte_msk <<= 1;
                    }
                    if (byte_cnt != 0)
                    {
                        xxBuf[xx_idx++] = byte_val;
                    }
                    //
                    break;
                //
                case def.WRITE_MULTIPLE_REGISTERS:
                    //
                    // Store item quantity.
                    xxBuf[xx_idx++] = (byte) (pkt.itm_qty >> 8);
                    xxBuf[xx_idx++] = (byte) pkt.itm_qty;
                    // Store number of bytes.
                    xxBuf[xx_idx++] = (byte) (pkt.itm_qty * 2);
                    // Store data.
                    for (i = 0; i < pkt.itm_qty; ++i)
                    {
                        xxBuf[xx_idx++] = (byte) (pkt.w[i] >> 8);
                        xxBuf[xx_idx++] = (byte) pkt.w[i];
                    }
                    //
                    break;
                //
                // --------------------------------------------------------------
                // Modbus RPC section, MASTER TRANSMIT.
                // --------------------------------------------------------------
                //
                case def.RD_REGS_STR:
                    //
                    // \/
                    // Format --> S F AA QQ
                    // Format <-- S F AA QQ B..B\0B..B\0...
                    //
                    // Store item quantity.
                    xxBuf[xx_idx++] = (byte) (pkt.itm_qty >> 8);
                    xxBuf[xx_idx++] = (byte) pkt.itm_qty;
                    //
                    break;
                //
                case def.WR_REGS_STR:
                    //
                    // Sending a series of null terminated strings.
                    // mex.itm_qty should be number of strings, not length of
                    // packet.
                    // Strings to send are in mex.s[ ].
                    //
                    // \/
                    // Format --> S F AA QQ B..B\0B..B\0...
                    // Format <-- S F AA
                    //
                    // Store item quantity.
                    xxBuf[xx_idx++] = (byte) (pkt.itm_qty >> 8);
                    xxBuf[xx_idx++] = (byte) pkt.itm_qty;
                    // Transfer string to byte array.
                    for (i = 0; i < pkt.itm_qty; ++i)
                    {
                        xx_idx = StringToBytes(xxBuf, xx_idx, pkt, i);
                    }
                    //
                    break;
                //
                case def.MRPC_FCN:
                    //
                    // \/
                    // Format --> S F AA QQ LLL...
                    // Format <-- S F AA QQ LLL...
                    //
                    // Store item quantity.
                    xxBuf[xx_idx++] = (byte) (pkt.itm_qty >> 8);
                    xxBuf[xx_idx++] = (byte) pkt.itm_qty;
                    // Store data.
                    for (i = 0; i < pkt.itm_qty; ++i)
                    {
                        xxBuf[xx_idx++] = (byte) (pkt.l[i] >> 24);
                        xxBuf[xx_idx++] = (byte) (pkt.l[i] >> 16);
                        xxBuf[xx_idx++] = (byte) (pkt.l[i] >> 8);
                        xxBuf[xx_idx++] = (byte) pkt.l[i];
                    }
                    //
                    break;
                //
                case def.RD_REGS_LIST:
                    //
                    // \/
                    // Format --> S F AA QQ RR RR...
                    // Format <-- S F AA QQ B..B\0B..B\0...
                    //
                    // Store item quantity.
                    xxBuf[xx_idx++] = (byte) (pkt.itm_qty >> 8);
                    xxBuf[xx_idx++] = (byte) pkt.itm_qty;
                    // Transfer register list.
                    for (i = 0; i < pkt.itm_qty; ++i)
                    {
                        xxBuf[xx_idx++] = (byte) (pkt.w[i] >> 8);
                        xxBuf[xx_idx++] = (byte) pkt.w[i];
                    }
                    //
                    break;
                //
                case def.RD_OBJ:
                    //
                    // \/
                    // Req : S F QQ IIII PP PP...
                    // Rsp : S F C c..c\0 c..c\0... (Success)
                    // Rsp : S F R (Failure)
                    //
                case def.WR_OBJ:
                    //
                    // \/
                    // Req : S F QQ IIII PP c..c\0 PP c..c\0...
                    // Rsp : S F C (Success)
                    // Rsp : S F R (Failure)
                    //
                    // NOTE: mex data storage:
                    // mex.l[0] = object instance
                    // mex.w[n] = property numbers
                    // mex.s[n] = property strings
                    //
                    // Note that mex.itm_adr actually has the quantity and
                    // index flag.
                    // Get the actual number of properties to read/write.
                    pkt.itm_qty = (short) (pkt.itm_adr & 0x7fff);
                    // device instance in network byte order.
                    xxBuf[xx_idx++] = (byte) (pkt.l[0] >> 24);
                    xxBuf[xx_idx++] = (byte) (pkt.l[0] >> 16);
                    xxBuf[xx_idx++] = (byte) (pkt.l[0] >> 8);
                    xxBuf[xx_idx++] = (byte) pkt.l[0];
                    // object instance in network byte order.
                    xxBuf[xx_idx++] = (byte) (pkt.l[1] >> 24);
                    xxBuf[xx_idx++] = (byte) (pkt.l[1] >> 16);
                    xxBuf[xx_idx++] = (byte) (pkt.l[1] >> 8);
                    xxBuf[xx_idx++] = (byte) pkt.l[1];
                    // For each property.
                    for (i = 0; i < pkt.itm_qty; ++i)
                    {
                        // Property number in network byte order.
                        xxBuf[xx_idx++] = (byte) (pkt.w[i] >> 8);
                        xxBuf[xx_idx++] = (byte) pkt.w[i];
                        // If writing.
                        if ((int) pkt.fcn == def.WR_OBJ)
                        {
                            // Property string.
                            xx_idx = StringToBytes(xxBuf, xx_idx, pkt, i);
                        }
                    }
                    break;
                //
                default:
                    //
                    // Throw error.
                    IOException e = new IOException(err.ILLEGAL_FUNCTION.name());
                    throw e;
            }
            // If this is Modbus/TCP.
            if (protocol == def.PROT_TCP)
            {
                // Get length for MBAP header and store it.
                frame_length = xx_idx - 6;
                xxBuf[4] = (byte) (frame_length >> 8);
                xxBuf[5] = (byte) frame_length;
            }
            // Create frame to transmit.
            frame_length = BinToTx(xx_idx, xxBuf, txBuf);
            // Send frame; throw exception if problem.
            try
            {
                dos.write(txBuf, 0, frame_length);
            } catch (IOException e)
            {
                throw e;
            }
            //
            // If this was a broadcast, we do not wait for a reply!!!
            //
            if (pkt.slv_adr == 0)
                return 0;
            // Wait for reply.
            try
            {
                frame_length = WaitPacket();
            } catch (IOException e)
            {
                throw e;
            }
            // Convert to binary.
            frame_length = RxToBin(frame_length, rxBuf, rxBuf);
            // Zero buffer index.
            rx_idx = 0;
            // If this is Modbus/TCP.
            if (protocol == def.PROT_TCP)
            {
                // Get transaction id.
                pkt.tran_id = (short) (rxBuf[rx_idx++] << 8);
                pkt.tran_id |= rxBuf[rx_idx++] & 0x00ff;
                // Get protocol id.
                pkt.prot_id = (short) (rxBuf[rx_idx++] << 8);
                pkt.prot_id |= rxBuf[rx_idx++] & 0x00ff;
                // Get mbap length field.
                pkt.mbap_len = (short) (rxBuf[rx_idx++] << 8);
                pkt.mbap_len |= rxBuf[rx_idx++] & 0x00ff;
            }
            // Store slave address.
            pkt.slv_adr = rxBuf[rx_idx++];
            // Store function (with possible error marker).
            pkt.fcn = rxBuf[rx_idx++];
            // If error response.
            if ((pkt.fcn & 0x80) != 0)
            {
                // Throw Modbus error.
                IOException e = new IOException(err.getName(rxBuf[rx_idx] & 0x0ff));
                throw e;
            }
            // Process return data.
            switch ((int) pkt.fcn)
            {
                case def.WRITE_SINGLE_COIL:
                case def.WRITE_SINGLE_REGISTER:
                case def.WRITE_MULTIPLE_COILS:
                case def.WRITE_MULTIPLE_REGISTERS:
                    //
                    // Break if writing data.
                    break;
                //
                case def.READ_COILS:
                case def.READ_DISCRETE_INPUTS:
                    //
                    // Skip byte count.
                    ++rx_idx;
                    // Set bitmask.
                    byte_msk = 1;
                    // For each item.
                    for (i = 0; i < pkt.itm_qty; ++i)
                    {
                        if ((rxBuf[rx_idx] & byte_msk) != 0)
                            pkt.b[i] = 1;
                        else
                            pkt.b[i] = 0;
                        if (byte_msk == 0x80)
                        {
                            ++rx_idx;
                            byte_msk = 1;
                        } else
                            byte_msk <<= 1;
                    }
                    //
                    break;
                //
                case def.READ_HOLDING_REGISTERS:
                case def.READ_INPUT_REGISTERS:
                    //
                    // Skip byte count.
                    ++rx_idx;
                    // Get register values.
                    for (i = 0; i < pkt.itm_qty; ++i)
                    {
                        pkt.w[i] = (short) (rxBuf[rx_idx++] << 8);
                        pkt.w[i] |= rxBuf[rx_idx++] & 0x00ff;
                    }
                    break;
                //
                // --------------------------------------------------------------
                // Modbus RPC section, MASTER RECEIVE.
                // --------------------------------------------------------------
                //
                case def.RD_REGS_STR:
                case def.RD_REGS_LIST:
                    //
                    // Receiving a series of null terminated strings.
                    //
                    // Format --> S F AA QQ
                    // \/
                    // Format <-- S F AA QQ B..B\0B..B\0...
                    //
                    // Skip address field.
                    rx_idx += 2;
                    // Store item quantity.
                    pkt.itm_qty = (short) (rxBuf[rx_idx++] << 8);
                    pkt.itm_qty |= rxBuf[rx_idx++] & 0x00ff;
                    // Transfer bytes to strings.
                    for (i = 0; i < pkt.itm_qty; ++i)
                    {
                        rx_idx = BytesToString(rxBuf, rx_idx, pkt, i);
                    }
                    //
                    break;
                //
                case def.WR_REGS_STR:
                    //
                    // No action required.
                    //
                    break;
                //
                case def.MRPC_FCN:
                    //
                    // Format --> S F AA QQ LLL...
                    // \/
                    // Format <-- S F AA QQ LLL...
                    //
                    // Skip address field.
                    rx_idx += 2;
                    // Store number of returned values.
                    pkt.itm_qty = (short) (rxBuf[rx_idx++] << 8);
                    pkt.itm_qty |= rxBuf[rx_idx++] & 0x00ff;
                    // Transfer returned values.
                    for (i = 0; i < pkt.itm_qty; ++i)
                    {
                        pkt.l[i] = (rxBuf[rx_idx++] << 24) & 0xff000000;
                        pkt.l[i] |= (rxBuf[rx_idx++] << 16) & 0x00ff0000;
                        pkt.l[i] |= (rxBuf[rx_idx++] << 8) & 0x0000ff00;
                        pkt.l[i] |= rxBuf[rx_idx++] & 0x000000ff;
                    }
                    //
                    break;
                //
                case def.RD_OBJ:
                    //
                    // Req : S F QQ IIII PP PP...
                    // \/
                    // Rsp : S F C c..c\0 c..c\0... (Success)
                    //
                case def.WR_OBJ:
                    //
                    // Req : S F QQ IIII PP c..c\0 PP c..c\0...
                    // \/
                    // Rsp : S F C (Success)
                    //
                    // NOTE: mex data storage:
                    // mex.l[0] = object instance
                    // mex.w[n] = property numbers
                    // mex.s[n] = property strings
                    //
                    // Nothing to do if writing.
                    if ((int) pkt.fcn == def.WR_OBJ)
                        break;
                    // Get number of strings.
                    pkt.itm_qty = (short) rxBuf[rx_idx++];
                    // For each string.
                    for (i = 0; i < pkt.itm_qty; ++i)
                    {
                        // Transfer the string.
                        rx_idx = BytesToString(rxBuf, rx_idx, pkt, i);
                    }
                    //
                    break;
                //
                default:
                    break;
            }
            // And return.
            return 0;
        }

        //
        // ----------------------------------------------------------------------
        //
        private void SetProtocol( byte mod_protocol )
        {
            protocol = mod_protocol;
        }

        //
        // ----------------------------------------------------------------------
        //
        private int GetProtocol()
        {
            return protocol;
        }

        //
        // ----------------------------------------------------------------------
        //
        // Constructor.
        private transfer( InputStream is , OutputStream os )
        {
            // Save input and output sources.
            dis = is;
            dos = os;
            // Allocate memory for buffers.
            rxBuf = new byte[BUFFER_SIZE];
            xxBuf = new byte[BUFFER_SIZE];
            txBuf = new byte[BUFFER_SIZE];
        }
    }
    
    public static class regOp32
    {
        public boolean  float_op;
        public boolean  big_endian;
        public short[]  w = new short[2];
        public int      i;
        public float    f;
    }

    public static class device
    {
        public final static int CONFIG_ADDRESS = 255;
        //
        //
        // Timeouts.
        //
        private final static int  TIMEOUT_CONNECT = 3000; // 3 sec.
        private final static int  TIMEOUT_RX      = 1000; // 1 sec.
        //
        // Modbus variables.
        //
        private final static byte SKT_BIT         = 0x01;
        private final static byte DIS_BIT         = 0x02;
        private final static byte DOS_BIT         = 0x04;
        //
        private Socket            skt;
        private byte              m_flags         = 0;
        //
        private InputStream       dis;
        private OutputStream      dos;
        //
        private packet            pkt;
        private transfer          trf;
        //
        private boolean           m_connected     = false;

        /**
         * Closes data input stream, data output stream, and socket.
         */
        public void Disconnect()
        {
            // If data input stream.
            if ((m_flags & DIS_BIT) != 0)
            {
                // Clear bit.
                m_flags &= ~DIS_BIT;
                // Close input stream.
                try
                {
                    dis.close();
                } catch (IOException e)
                {
                }
            }
            // If data output stream.
            if ((m_flags & DOS_BIT) != 0)
            {
                // Clear bit.
                m_flags &= ~DOS_BIT;
                // Close output stream.
                try
                {
                    dos.flush();
                    dos.close();
                } catch (IOException e)
                {
                }
            }
            // If socket.
            if ((m_flags & SKT_BIT) != 0)
            {
                // Clear bit.
                m_flags &= ~SKT_BIT;
                // Close socket.
                try
                {
                    skt.close();
                } catch (IOException e)
                {
                }
            }
            // Clear flag.
            m_connected = false;
        }

        /**
         * Connect to the target and setup to use the specified protocol.
         * 
         * @param DottedIpAddress
         *            = IP Address of target.
         * @param port
         *            = Host port number.
         * @param protocol
         *            = mod.def.PROT_XXX protocol.
         * @return = 0 if success, -1 otherwise.
         */
        public int Connect( String DottedIpAddress , int port , int protocol )
        {
            InetSocketAddress addr;

            // Create our socket and set flag.
            skt = new Socket();
            m_flags |= SKT_BIT;

            // Bind to a local ephemeral port.
            try
            {
                skt.bind(null);
            } catch (IOException e)
            {
                Disconnect();
                return -1;
            }

            // Get socket address.
            addr = new InetSocketAddress(DottedIpAddress, port);

            // If port is unresolved.
            if (addr.isUnresolved())
                return -1;

            // Connect with timeout.
            try
            {
                skt.connect(addr, TIMEOUT_CONNECT);
            } catch (IOException e)
            {
                Disconnect();
                return -1;
            }

            // Putting in a timeout because the receive loop could hang
            // waiting for a character.
            try
            {
                skt.setSoTimeout(TIMEOUT_RX);
            } catch (IOException e)
            {
                Disconnect();
                return -1;
            }

            // Get stream for output.
            try
            {
                dos = skt.getOutputStream();
            } catch (IOException e)
            {
                Disconnect();
                return -1;
            }
            // Set flag bit.
            m_flags |= DOS_BIT;

            // Get stream for input.
            try
            {
                dis = skt.getInputStream();
            } catch (IOException e)
            {
                Disconnect();
                return -1;
            }
            // Set flag bit.
            m_flags |= DIS_BIT;

            // Create our modbus exchange objects.
            pkt = new packet();
            trf = new transfer(dis, dos);
            // Assign protocol.
            trf.SetProtocol((byte) protocol);

            // Set the flag.
            m_connected = true;

            // And return.
            return 0;
        }

        /**
         * Checks to see if tcp connection has been established.
         * 
         * @return true=connected, false=not connected.
         */
        public boolean IsConnected()
        {
            return m_connected;
        }

        //
        // ----------------------------------------------------------------------
        //
        public int SetTimeout( int msTimeout )
        {
            try
            {
                if ( msTimeout == -1 )
                    skt.setSoTimeout(TIMEOUT_RX);
                else
                    skt.setSoTimeout(msTimeout);
            } catch (IOException e)
            {
                Disconnect();
                return -1;
            }
            return 0;
        }

        //
        // ----------------------------------------------------------------------
        //
        public void SetProtocol( int protocol )
        {
            switch (protocol)
            {
                case def.PROT_DF1:
                case def.PROT_TCP:
                    trf.SetProtocol((byte) protocol);
                    break;
                default:
                    break;
            }
        }

        //
        // ----------------------------------------------------------------------
        //
        public int GetProtocol()
        {
            return trf.GetProtocol();
        }

        //
        // ----------------------------------------------------------------------
        /**
         * Returns def.ErrStr[mex.b[0]].
         */
        public String GetError()
        {
            return err.getName(pkt.b[0] & 0x0ff);
        }

        public int ReadCoil( int slave , int reg ) throws IOException
        {
            pkt.slv_adr = (byte) slave;
            pkt.fcn = (byte) mod_def.READ_COILS;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = 1;
            try
            {
                trf.MasterTx(pkt);
                if (pkt.b[0] > 0)
                    return 1;
                else
                    return 0;
            } catch (IOException e)
            {
                throw e;
            }
        }

        public int WriteCoil( int slave , int reg , int val ) throws IOException
        {
            pkt.slv_adr = (byte) slave;
            pkt.fcn = (byte) mod_def.WRITE_SINGLE_COIL;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = 1;
            pkt.b[0] = (byte) val;
            try
            {
                return trf.MasterTx(pkt);
            } catch (IOException e)
            {
                throw e;
            }
        }
        
        public int ReadInp(int slave , int reg) throws IOException
        {
            pkt.slv_adr = (byte) slave;
            pkt.fcn = (byte) mod_def.READ_DISCRETE_INPUTS;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = 1;
            try
            {
                trf.MasterTx(pkt);
                if ((pkt.b[0] & 1) > 0)
                {
                    return 1;
                }
                else
                {
                    return 0;
                }
            }
            catch (IOException e)
            {
                throw e;
            }
        }
        
        public int ReadRegInp16(int slave, int reg) throws IOException
        {
            pkt.slv_adr = (byte) slave;
            pkt.fcn = (byte) mod_def.READ_INPUT_REGISTERS;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = 1;
            try
            {
                trf.MasterTx(pkt);
                return pkt.w[0] & 0x0ffff;
            }
            catch (IOException e)
            {
                throw e;
            }
        }
        /**
         * Reads an unsigned 16-bit value. Can be used with any Modbus device.
         * 
         * @param slave = Modbus slave address.
         * @param reg = Register number.
         * @return = Unsigned 16-bit value if success, throws IOException otherwise.
         * @throws IOException
         */
        public int ReadReg16(int slave, int reg) throws IOException
        {
            pkt.slv_adr = (byte) slave;
            pkt.fcn = (byte) mod_def.READ_HOLDING_REGISTERS;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = 1;
            try
            {
                trf.MasterTx(pkt);
                return pkt.w[0] & 0x0ffff;
            }
            catch (IOException e)
            {
                throw e;
            }
        }
        
        /**
         * Reads unsigned 16-bit values into an array of short integers.
         * Can be used with any Modbus device.
         * 
         * @param slave = Modbus slave address.
         * @param reg = Register number.
         * @param dst = Array of short integers.
         * @param qty = Number of registers to read.
         * @return = Unsigned 16-bit value if success, throws IOException otherwise.
         * @throws IOException
         */
        public int ReadReg16(int slave, int reg , short dst[] , int qty) throws IOException
        {
            int         i;
            
            pkt.slv_adr = (byte) slave;
            pkt.fcn = (byte) mod_def.READ_HOLDING_REGISTERS;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = (short) qty;
            try
            {
                trf.MasterTx(pkt);
                for ( i = 0 ; i < qty ; ++i )
                {
                    dst[i] = pkt.w[i];
                }
                return 0;
            }
            catch (IOException e)
            {
                throw e;
            }
        }
        
        /**
         * Writes an unsiged 16-bit value. Can be used with any Modbus device.
         * 
         * @param slave= Modbus slave address.
         * @param reg = Register number.
         * @param value = 16-bit value to write.
         * @return 0=success, throws IOException otherwise.
         * @throws IOException
         */
        public int WriteReg16(int slave, int reg, int value) throws IOException
        {
            pkt.slv_adr = (byte) slave;
            pkt.fcn = (byte) mod_def.WRITE_MULTIPLE_REGISTERS;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = 1;
            pkt.w[0] = (short) value;
            try
            {
                return trf.MasterTx(pkt);
            }
            catch (IOException e)
            {
                throw e;
            }
        }
        
        /**
         * Writes an array of unsiged 16-bit values. Can be used with any Modbus device.
         * Returns 0 if success, -1 otherwise. Member 'm_error' is set to 0 if
         * success, Modbus error code if problem.
         * 
         * @param slave= Modbus slave address.
         * @param reg = Register number.
         * @param value = 16-bit value to write.
         * @return 0=success, throws IOException otherwise.
         * @throws IOException
         */
        public int WriteReg16(int slave, int reg, int value[] , int qty) throws IOException
        {
            int             i;
            
            pkt.slv_adr = (byte) slave;
            pkt.fcn = (byte) mod_def.WRITE_MULTIPLE_REGISTERS;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = (short)qty;
            for ( i = 0 ; i < pkt.itm_qty ; ++i )
            {
                pkt.w[i] = (short)value[i];
            }
            try
            {
                return trf.MasterTx(pkt);
            }
            catch (IOException e)
            {
                throw e;
            }
        }
        
        public int RegInpPair(int slave, int reg, boolean write) throws IOException
        {
            pkt.slv_adr = (byte) slave;
            if (write)
                return -1;
            else
                pkt.fcn = (byte) mod_def.READ_INPUT_REGISTERS;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = 2;
            try
            {
                return trf.MasterTx(pkt);
            }
            catch (IOException e)
            {
                throw e;
            }
        }
        /**
         * Reads or writes two consecutive registers.
         * 
         * @param slave = Modbus slave address.
         * @param reg = First register number (1-based).
         * @param write = true if write operation.
         * @return 0=success, throws IOException otherwise.
         * @throws IOException
         */
        public int RegPair(int slave, int reg, boolean write) throws IOException
        {
            pkt.slv_adr = (byte) slave;
            if (write)
                pkt.fcn = (byte) mod_def.WRITE_MULTIPLE_REGISTERS;
            else
                pkt.fcn = (byte) mod_def.READ_HOLDING_REGISTERS;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = 2;
            try
            {
                return trf.MasterTx(pkt);
            }
            catch (IOException e)
            {
                throw e;
            }
        }
        
        public int ReadRegPair(int slave, int reg, regOp32 rop) throws IOException
        {
            int val;

            try
            {
                RegPair(slave, reg, false);
            }
            catch (IOException e)
            {
                throw e;
            }
            if (rop.big_endian)
            {
                val = pkt.w[0] & 0x0000ffff;
                val <<= 16;
                val |= pkt.w[1] & 0x0000ffff;
            }
            else
            {
                val = pkt.w[1] & 0x0000ffff;
                val <<= 16;
                val |= pkt.w[0] & 0x0000ffff;
            }
            if ( rop.float_op )
                rop.f = Float.intBitsToFloat(val);
            else
                rop.i = val;
            return 0;
        }

        public int ReadRegInpPair(int slave, int reg, regOp32 rop) throws IOException
        {
            int val;

            try
            {
                RegInpPair(slave, reg, false);
            }
            catch (IOException e)
            {
                throw e;
            }
            if (rop.big_endian)
            {
                val = pkt.w[0] & 0x0000ffff;
                val <<= 16;
                val |= pkt.w[1] & 0x0000ffff;
            }
            else
            {
                val = pkt.w[1] & 0x0000ffff;
                val <<= 16;
                val |= pkt.w[0] & 0x0000ffff;
            }
            if ( rop.float_op )
                rop.f = Float.intBitsToFloat(val);
            else
                rop.i = val;
            return 0;
        }

        public int WriteRegPair(int slave, int reg, regOp32 rop) throws IOException
        {
            int val;

            if (rop.float_op)
                val = Float.floatToRawIntBits(rop.f);
            else
                val = rop.i;
            if (rop.big_endian)
            {
                pkt.w[0] = (short) (val >> 16);
                pkt.w[1] = (short) val;
            }
            else
            {
                pkt.w[0] = (short) val;
                pkt.w[1] = (short) (val >> 16);
            }
            try
            {
                return RegPair(slave, reg, true);
            }
            catch (IOException e)
            {
                throw e;
            }
        }

        public int ReadRegInt( int slv , int reg ) throws IOException
        {
            pkt.slv_adr = (byte) slv;
            pkt.fcn = (byte) def.RD_REGS_STR;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = 1;
            try
            {
                trf.MasterTx(pkt);
                try
                {
                    return Integer.decode(pkt.s[0]);
                } catch (NumberFormatException ne)
                {
                    IOException e = new IOException(err.ILLEGAL_DATA_VALUE.name());
                    throw e;
                }
            } catch (IOException e)
            {
                throw e;
            }
        }

        public String ReadRegStr( int slv , int reg ) throws IOException
        {
            pkt.slv_adr = (byte) slv;
            pkt.fcn = (byte) def.RD_REGS_STR;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = 1;
            try
            {
                trf.MasterTx(pkt);
                return pkt.s[0];
            } catch (IOException e)
            {
                throw e;
            }
        }

        public void ReadRegStr( int slv , int reg , String[] dst , int qty ) throws IOException
        {
            int         i;
            
            pkt.slv_adr = (byte) slv;
            pkt.fcn = (byte) def.RD_REGS_STR;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = (short)qty;
            try
            {
                trf.MasterTx(pkt);
                for ( i = 0 ; i < qty ; ++i )
                {
                    dst[i] = pkt.s[i];
                }
            } catch (IOException e)
            {
                throw e;
            }
        }
        
        public void ReadRegList( int slv , int[] reg , String[] dst , int qty ) throws IOException
        {
            int         i;
            
            pkt.slv_adr = (byte) slv;
            pkt.fcn = (byte) def.RD_REGS_LIST;
            pkt.itm_adr = 0;
            pkt.itm_qty = (short)qty;
            for ( i = 0 ; i < qty ; ++i )
            {
                pkt.w[i] = (short)reg[i];
            }
            try
            {
                trf.MasterTx(pkt);
                for ( i = 0 ; i < qty ; ++i )
                {
                    dst[i] = pkt.s[i];
                }
            } catch (IOException e)
            {
                throw e;
            }
        }

        public int WriteReg(int slv , int reg , String val ) throws IOException
        {
            pkt.slv_adr = (byte) slv;
            pkt.fcn = (byte) def.WR_REGS_STR;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = 1;
            pkt.s[0] = val;
            try
            {
                return trf.MasterTx(pkt);
            } catch (IOException e)
            {
                throw e;
            }
        }

        public int WriteReg(int slv , int reg , int val ) throws IOException
        {
            pkt.slv_adr = (byte) slv;
            pkt.fcn = (byte) def.WR_REGS_STR;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = 1;
            pkt.s[0] = Integer.toString(val);
            try
            {
                return trf.MasterTx(pkt);
            } catch (IOException e)
            {
                throw e;
            }
        }

        /**
         * Returns the integer value of an object property.
         * 
         * @param instance
         *            = Object instance
         * @param property
         *            = Object property
         * @return Property value if success, -1 if number format exception
         *         generated while converting returned string to integer.
         * @throws IOException
         */
        public int ReadObjInt( int devInstance, int instance , int property ) throws IOException
        {
            pkt.slv_adr = (byte) CONFIG_ADDRESS;
            pkt.fcn = (byte) def.RD_OBJ;
            // NOTE: itm_adr contains index flag and quantity;
            //       itm_qty is not used.
            pkt.itm_adr = 1;
            pkt.itm_qty = 1;
            pkt.l[0] = devInstance;
            pkt.l[1] = instance;
            pkt.w[0] = (short) property;
            try
            {
                trf.MasterTx(pkt);
                try
                {
                    return Integer.decode(pkt.s[0]);
                } catch (NumberFormatException ne)
                {
                    IOException e = new IOException(err.ILLEGAL_DATA_VALUE.name());
                    throw e;
                }
            } catch (IOException e)
            {
                throw e;
            }
        }

        /**
         * Returns the integer value of an object property using
         * an indexed read.
         * 
         * @param dev_instance
         *            = device index
         * @param instance
         *            = Object index
         * @param property
         *            = Object property
         * @return Property value if success, -1 if number format exception
         *         generated while converting returned string to integer.
         * @throws IOException
         */
        public int ReadObjIntX( int devIndex, int index , int property ) throws IOException
        {
            pkt.slv_adr = (byte) CONFIG_ADDRESS;
            pkt.fcn = (byte) def.RD_OBJ;
            // NOTE: itm_adr contains index flag and quantity;
            //       itm_qty is not used.
            pkt.itm_adr = (short)0x8001;
            pkt.itm_qty = 1;
            pkt.l[0] = devIndex;
            pkt.l[1] = index;
            pkt.w[0] = (short) property;
            try
            {
                trf.MasterTx(pkt);
                try
                {
                    return Integer.decode(pkt.s[0]);
                } catch (NumberFormatException ne)
                {
                    IOException e = new IOException(err.ILLEGAL_DATA_VALUE.name());
                    throw e;
                }
            } catch (IOException e)
            {
                throw e;
            }
        }

        public String ReadObjStr( int devInstance, int instance , int property ) throws IOException
        {
            pkt.slv_adr = (byte) CONFIG_ADDRESS;
            pkt.fcn = (byte) def.RD_OBJ;
            // NOTE: itm_adr contains index flag and quantity;
            //       itm_qty is not used.
            pkt.itm_adr = 1;
            pkt.itm_qty = 1;
            pkt.l[0] = devInstance;
            pkt.l[1] = instance;
            pkt.w[0] = (short) property;
            try
            {
                trf.MasterTx(pkt);
                return pkt.s[0];
            } catch (IOException e)
            {
                throw e;
            }
        }

        public String ReadObjStrX( int devIndex, int index , int property ) throws IOException
        {
            pkt.slv_adr = (byte) CONFIG_ADDRESS;
            pkt.fcn = (byte) def.RD_OBJ;
            // NOTE: itm_adr contains index flag and quantity;
            //       itm_qty is not used.
            pkt.itm_adr = (short)0x8001;
            pkt.itm_qty = 1;
            pkt.l[0] = devIndex;
            pkt.l[1] = index;
            pkt.w[0] = (short) property;
            try
            {
                trf.MasterTx(pkt);
                return pkt.s[0];
            } catch (IOException e)
            {
                throw e;
            }
        }

        public int WriteObj( int devInstance, int instance , int property , int value ) throws IOException
        {
            pkt.slv_adr = (byte) CONFIG_ADDRESS;
            pkt.fcn = (byte) def.WR_OBJ;
            pkt.itm_adr = 1;
            pkt.itm_qty = 1;
            pkt.l[0] = devInstance;
            pkt.l[1] = instance;
            pkt.w[0] = (short) property;
            pkt.s[0] = Integer.toString(value);
            try
            {
                return trf.MasterTx(pkt);
            } catch (IOException e)
            {
                throw e;
            }
        }

        public int WriteObj( int devInstance, int instance , int property , String value ) throws IOException
        {
            pkt.slv_adr = (byte) CONFIG_ADDRESS;
            pkt.fcn = (byte) def.WR_OBJ;
            pkt.itm_adr = 1;
            pkt.itm_qty = 1;
            pkt.l[0] = devInstance;
            pkt.l[1] = instance;
            pkt.w[0] = (short) property;
            pkt.s[0] = value;
            try
            {
                return trf.MasterTx(pkt);
            } catch (IOException e)
            {
                throw e;
            }
        }

        public int ReadCfgInt( int reg ) throws IOException
        {
            pkt.slv_adr = (byte) CONFIG_ADDRESS;
            pkt.fcn = (byte) def.RD_REGS_STR;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = 1;
            try
            {
                trf.MasterTx(pkt);
                try
                {
                    return Integer.decode(pkt.s[0]);
                } catch (NumberFormatException ne)
                {
                    IOException e = new IOException(err.ILLEGAL_DATA_VALUE.name());
                    throw e;
                }
            } catch (IOException e)
            {
                throw e;
            }
        }

        public String ReadCfgStr( int reg ) throws IOException
        {
            pkt.slv_adr = (byte) CONFIG_ADDRESS;
            pkt.fcn = (byte) def.RD_REGS_STR;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = 1;
            try
            {
                trf.MasterTx(pkt);
                return pkt.s[0];
            } catch (IOException e)
            {
                throw e;
            }
        }

        public int WriteCfg( int reg , String val ) throws IOException
        {
            pkt.slv_adr = (byte) CONFIG_ADDRESS;
            pkt.fcn = (byte) def.WR_REGS_STR;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = 1;
            pkt.s[0] = val;
            try
            {
                return trf.MasterTx(pkt);
            } catch (IOException e)
            {
                throw e;
            }
        }

        public int WriteCfg( int reg , int val ) throws IOException
        {
            pkt.slv_adr = (byte) CONFIG_ADDRESS;
            pkt.fcn = (byte) def.WR_REGS_STR;
            pkt.itm_adr = (short) reg;
            pkt.itm_qty = 1;
            pkt.s[0] = Integer.toString(val);
            try
            {
                return trf.MasterTx(pkt);
            } catch (IOException e)
            {
                throw e;
            }
        }
    }

    //
    // -------------------------------------------------------------------------
    //
    // STATIC FUNCTIONS.
    //
    // -------------------------------------------------------------------------
    //
    /**
     * Here to return a complete register index with the channel in the
     * thousands place.
     * 
     * @param chn
     *            = BAS 0-based channel number.
     * @param reg
     *            = BAS 1-based indexed register.
     * @return = indexed channel number.
     */
    public static int RegIdx( int chn , int reg )
    {
        return reg + ((chn + 1) * 1000);
    }
}
