//
// Copyright (c) 2009 Tridium, Inc.
// Licensed under the Academic Free License version 3.0
//
// History:
//   03 June 09  Matthew Giannini  Creation
//
package sedonac.steps;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.InputStream;
import java.io.PrintWriter;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Properties;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import sedona.Env;
import sedona.platform.NativeManifest;
import sedona.platform.PlatformManifest;
import sedona.util.FileUtil;
import sedona.xml.XWriter;
import sedonac.Compiler;
import sedonac.CompilerStep;
import sedonac.Location;
import sedonac.ir.IrKit;
import sedonac.ir.IrMethod;
import sedonac.namespace.NativeId;
import sedonac.namespace.Slot;
import sedonac.platform.PlatformDef;
import sedonac.util.CStrGen;
import sedonac.util.VarResolver;

/**
 * Creates and stages the platform manifest and sedonaPlatform.h header file.
 */
public class StagePlatform extends CompilerStep
{
  public StagePlatform(Compiler compiler)
  {
    super(compiler);
  }

  public void run()
  {
    log.info("  StagePlatform");
    makePlatformManifest(compiler.platform);
    writeCHeaders();
    writeManifest();
    writeCPlatformManifest();
    quitIfErrors();
  }

  private void makePlatformManifest(PlatformDef platDef)
  {
    // platformManifest root attributes
    manifest = new PlatformManifest();
    manifest.id         = platDef.idPattern;
    manifest.vendor     = platDef.vendor;
    manifest.endian     = platDef.endian;
    manifest.blockSize  = platDef.blockSize;
    manifest.refSize    = platDef.refSize;
    manifest.armDouble  = platDef.armDouble;
    manifest.debug      = platDef.debug;
    manifest.test       = platDef.test;

    // native kits
    manifest.nativeKits = platDef.nativeKits;

    // native methods
    Slot[] nativeMethods = findNativeMethods();
    manifest.nativeMethods = new NativeManifest[nativeMethods.length];
    for (int i=0; i<nativeMethods.length; ++i)
      manifest.nativeMethods[i] = toNativeManifest(manifest, nativeMethods[i]);
    Arrays.sort(manifest.nativeMethods, new NativeComparator());

    // manifest includes
    for (int i=0; i<platDef.manifestIncludes.length; ++i)
      manifest.manifestIncludes.addContent(platDef.manifestIncludes[i]);

    resolvePlatformId();
  }

  private Slot[] findNativeMethods()
  {
    for (int i=0; i<compiler.kits.length; ++i)
    {
      IrKit kit = compiler.kits[i];
      if (!kit.manifest.hasNatives) continue;
      for (int j=0; j<kit.types.length; ++j)
      {
        Slot[] slots = kit.types[j].slots();
        for (int k=0; k<slots.length; ++k)
        {
          if (slots[k].isNative())
            natives.put(slots[k].qname(), slots[k]);
        }
      }
    }
    return (Slot[])natives.values().toArray(new Slot[natives.size()]);
  }

  private NativeManifest toNativeManifest(PlatformManifest manifest, Slot nativeSlot)
  {
    IrMethod method = (IrMethod)nativeSlot;
    return new NativeManifest(manifest, method.qname(), method.nativeId.string);
  }

  private void resolvePlatformId()
  {
    if (manifest.id == null) return;

    Properties vars = new Properties();
    vars.put("stage.nativeChecksum", calcNativeCksum());
    try
    {
      manifest.id = (new VarResolver(vars)).resolve(manifest.id);
    }
    catch (Exception e)
    {
      throw err(e.getMessage(), "id attribute");
    }
  }

  private String calcNativeCksum()
  {
    // Assumes natives have already been ordered.
    java.util.zip.CRC32 crc = new java.util.zip.CRC32();
    for (int i=0; i<manifest.nativeMethods.length; ++i)
    {
      NativeManifest nm = manifest.nativeMethods[i];
      crc.update(nm.qname.getBytes());
      crc.update(toParamsString((IrMethod)natives.get(nm.qname)).getBytes());
      crc.update(nm.nativeId.getBytes());
    }
    return Integer.toHexString((int)crc.getValue());
  }

  /**
   * Get a String of the native params for calculating the native checksum.
   */
  private String toParamsString(IrMethod nm)
  {
    StringBuffer sb = new StringBuffer();
    for (int i=0; i<nm.params.length; ++i)
    {
      if (i>0) sb.append(',');
      sb.append(nm.params[i]);
    }
    return sb.append(':').append(nm.returnType()).toString();
  }

  private void writeCHeaders()
  {
    if (manifest.id == null) return;

    File file = new File(compiler.outDir, "sedonaPlatform.h");
    log.debug("    Writing [ " + file + " ]");
    try
    {
      PrintWriter out = new PrintWriter(new FileWriter(file));
      out.println("//");
      out.println("// Generated by sedonac " + Env.version);
      out.println("// " + Env.timestamp());
      out.println("//");
      out.println();
      out.println("#ifndef SEDONAC_SEDONA_PLATFORM_H");
      out.println("#define SEDONAC_SEDONA_PLATFORM_H");
      out.println();
      out.println("#define PLATFORM_ID \"" + manifest.id + "\"");
      out.println();
      out.println("#endif");
      out.close();
    }
    catch (Exception e)
    {
      throw err("Cannot write sedonaPlatform.h", new Location(file), e);
    }
  }

  private void writeManifest()
  {
    File parDir = new File(compiler.outDir, ".par");
    manifestFile = new File(parDir, "platformManifest.xml");
    log.debug("    Writing [ " + manifestFile + " ]");
    try
    {
      XWriter out = new XWriter(manifestFile);
      manifest.encodeXml(out);
      out.close();
    }
    catch (Exception e)
    {
      throw err("Cannot write platformManifest.xml", new Location(manifestFile), e);
    }
  }
  
  private void writeCPlatformManifest()
  {
    if (!compiler.platform.embedManifest)
      return;
    try
    {
      // the platform manifest must be zipped when embedded in an svm
      InputStream in = new FileInputStream(manifestFile);
      File zipFile = new File(compiler.outDir, "pm.zip");
      ZipOutputStream zout = new ZipOutputStream(new FileOutputStream(zipFile));
      zout.putNextEntry(new ZipEntry("platformManifest.xml"));
      FileUtil.pipe(in, zout);
      in.close();
      zout.closeEntry();
      zout.close();

      File out = new File(compiler.outDir, "platformManifest.c");
      CStrGen.toCFile(zipFile, out, "pm");
    }
    catch (Exception e)
    {
      e.printStackTrace();
      throw err("Failed to write platformManifest.c");
    }
  }

  private static class NativeComparator implements Comparator
  {
    public int compare(Object o1, Object o2)
    {
      NativeId id1 = NativeId.parse(null, ((NativeManifest)o1).nativeId);
      NativeId id2 = NativeId.parse(null, ((NativeManifest)o2).nativeId);
      if (id1.equals(id2)) throw new IllegalStateException("Duplicate nativeIds: " + id1 + " " + id2);
      final int kitCompare = id1.kitId - id2.kitId;
      return (kitCompare == 0) ? id1.methodId - id2.methodId : kitCompare;
    }
  }

  PlatformManifest manifest;
  File manifestFile;
  /** qname -> IrMethod */
  HashMap natives = new HashMap();

}
